unit sample2;

{ based on Swiss Ephemeris sample code for Delphi 2
  extensions by Robert Amlung, Strasbourg }

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls;

type
  TForm1 = class(TForm)
    Label1: TLabel;
    Label2: TLabel;
    Label3: TLabel;
    Label4: TLabel;
    Label5: TLabel;
    button_doit: TButton;
    button_exit: TButton;
    day: TEdit;
    month: TEdit;
    year: TEdit;
    outfld: TMemo;
    housesfld: TMemo;
    hour: TEdit;
    minute: TEdit;
    second: TEdit;
    combo_utet: TComboBox;
    lon_deg: TEdit;
    lon_min: TEdit;
    lon_sec: TEdit;
    Combo_EW: TComboBox;
    lat_deg: TEdit;
    lat_min: TEdit;
    lat_sec: TEdit;
    Combo_NS: TComboBox;
    combo_hsys: TComboBox;
    combo_Birthplanet: TComboBox;
    LabelBirthplanet: TLabel;
    combo_Transits: TComboBox;
    LabelTransite: TLabel;
    button_load: TButton;
    button_save: TButton;
    Person: TEdit;
    OpenDialog: TOpenDialog;
    Bevel1: TBevel;
    button_now: TButton;
    procedure FormCreate(Sender: TObject);
    procedure button_doitClick(Sender: TObject);
    procedure button_exitClick(Sender: TObject);
    procedure button_loadClick(Sender: TObject);
    procedure button_saveClick(Sender: TObject);
    procedure button_nowClick(Sender: TObject);
  private
    { Private-Deklarationen }
  public
    { Public-Deklarationen }
  end;

var
  Form1: TForm1;

implementation

const

   DataDir : string = 'd:\sweph\data\';
   EphDir : array[0..80] of char = 'd:\sweph\ephe\';
   TimeDiff : integer = -1;

   SE_ECL_NUT           = -1;
   SE_SUN               = 0;
   SE_MOON              = 1;
   SE_MERCURY           = 2;
   SE_VENUS             = 3;
   SE_MARS              = 4;
   SE_JUPITER           = 5;
   SE_SATURN            = 6;
   SE_URANUS            = 7;
   SE_NEPTUNE           = 8;
   SE_PLUTO             = 9;
   SE_MEAN_NODE         = 10;
   SE_TRUE_NODE         = 11;
   SE_MEAN_APOG         = 12;
   SE_OSCU_APOG         = 13;
   SE_EARTH             = 14;
   SE_CHIRON            = 15;
   SE_PHOLUS            = 16;
   SE_NESSUS            = 17066;
   SE_CERES             = 17;
   SE_PALLAS            = 18;
   SE_JUNO              = 19;
   SE_VESTA             = 20;

   SE_NPLANETS : integer =     21;
   SE_AST_OFFSET : integer =   10000;
   SE_FICT_OFFSET : integer =  40;
   SE_NFICT_ELEM : integer =   15;

   { Hamburger or Uranian "planets" }
   SE_CUPIDO : integer =       	40;
   SE_HADES : integer =        	41;
   SE_ZEUS : integer =         	42;
   SE_KRONOS : integer =       	43;
   SE_APOLLON : integer =      	44;
   SE_ADMETOS : integer =      	45;
   SE_VULKANUS : integer =     	46;
   SE_POSEIDON : integer =     	47;
   { other ficticious bodies }
   SE_ISIS : integer =         	48;
   SE_NIBIRU : integer =       	49;
   SE_HARRINGTON : integer =           50;
   SE_NEPTUNE_LEVERRIER : integer =    51;
   SE_NEPTUNE_ADAMS : integer =        52;
   SE_PLUTO_LOWELL : integer =         53;
   SE_PLUTO_PICKERING : integer =      54;

   {  flag bits for parameter iflag in function swe_calc()
   The flag bits are defined in such a way that iflag = 0 delivers what one
   usually wants:
     - the default ephemeris (SWISS) is used,
     - apparent geocentric positions referring to the true equinox of date are returned.
     If not only coordinates, but also speed values are required, use iflag = SEFLG_SPEED.
   }
   SEFLG_JPLEPH : Longint =    	1;       { use JPL ephemeris }
   SEFLG_SWIEPH : Longint =    	2;       { use SWISSEPH ephemeris }
   SEFLG_MOSEPH : Longint =    	4;       { use Moshier ephemeris }
   SEFLG_HELCTR : Longint =        8;      { return heliocentric position }
   SEFLG_TRUEPOS : Longint =       16;    { return true positions, not apparent }
   SEFLG_J2000 : Longint =         32;    { no precession, i.e. give J2000 equinox }
   SEFLG_NONUT : Longint =         64;    { no nutation, i.e. mean equinox of date }
   SEFLG_SPEED : Longint =         256;    { high precision speed (analytical computation) }
   SEFLG_NOGDEFL : Longint =       512;    { turn off gravitational deflection }
   SEFLG_NOABERR : Longint =       1024;   { turn off 'annual' aberration of light }
   SEFLG_EQUATORIAL : Longint =    (2*1024);   { equatorial positions are wanted }
   SEFLG_XYZ : Longint =           (4*1024);   { cartesian, not polar, coordinates are wanted}
   SEFLG_RADIANS : Longint =       (8*1024);   { coordinates are wanted in radians, not degrees }
   SEFLG_BARYCTR : Longint =       (16*1024);  { barycentric positions }
   SEFLG_TOPOCTR : Longint =       (32*1024);  { topocentric positions }

   se_jul_cal    = 0;
   se_greg_cal   = 1;

var
  iday, imonth, iyear, ipl, ihour, imin, isec: Integer;
  xx    : Array[0..5] of double;
  serr :  Array[0..255] of Char;

function swe_calc(tjd          : double;    {Julian day number}
                   ipl         : Integer;   {planet number}
                   iflag       : Longint;   {flag bits}
                   var xx      : double;    {first of 6 doubles}
                   sErr        : PChar      {Error-String}
): Longint; stdcall; far; external 'swedll32.dll' name '_swe_calc@24';

procedure swe_close; stdcall; far; external 'swedll32.dll' name '_swe_close@0';

procedure swe_cotrans(var xpin  : double;
                   var xpout    : double;
                   eps          : double
); stdcall; far; external 'swedll32.dll' name '_swe_cotrans@16';

procedure swe_cotrans_sp(var xpin  : double;
                   var xpout    : double;
                   eps          : double
); stdcall; far; external 'swedll32.dll' name '_swe_cotrans_sp@16';

function swe_csnorm(p: Longint
): Longint; stdcall; far; external 'swedll32.dll' name '_swe_csnorm@4';

function swe_csroundsec(x: Longint
): Longint; stdcall; far; external 'swedll32.dll' name '_swe_csroundsec@4';

function swe_cs2degstr(t: Longint;
                   s     : pchar
): pchar; stdcall; far; external 'swedll32.dll' name '_swe_cs2degstr@8';

function swe_cs2lonlatstr(t: Longint;
                   pch   : char;
                   mch   : char;
                   s     : pchar
): pchar; stdcall; far; external 'swedll32.dll' name '_swe_cs2lonlatstr_d@16';

function swe_cs2timestr(t: Longint;
                   sep   : integer;
                   suppresszero : integer;
                   s     : pchar
): pchar; stdcall; far; external 'swedll32.dll' name '_swe_cs2timestr@16';

function swe_date_conversion(year: integer;
                   month       : integer;
                   day         : integer;
                   utime       : double;
                   c           : char;      {'g'regorian or 'j'ulian}
                   var tjd     : double
): integer; stdcall; far; external 'swedll32.dll' name '_swe_date_conversion@28';

function swe_day_of_week(var tjd: double
): integer; stdcall; far; external 'swedll32.dll' name '_swe_day_of_week@8';

function swe_degnorm(x: double
): double; stdcall; far; external 'swedll32.dll' name '_swe_degnorm@8';

function swe_deltat(tjd  : double
): double; stdcall; far; external 'swedll32.dll' name '_swe_deltat@8';

function swe_difcsn(p1: Longint;
                   p2 : Longint
): Longint; stdcall; far; external 'swedll32.dll' name '_swe_difcsn@8';

function swe_difcs2n(p1: Longint;
                   p2 : Longint
): Longint; stdcall; far; external 'swedll32.dll' name '_swe_difcs2n@8';

function swe_difdegn(p1 : double;
                   p2     : double
): double; stdcall; far; external 'swedll32.dll' name '_swe_difdegn@16';

function swe_difdeg2n(p1 : double;
                   p2     : double                   
): double; stdcall; far; external 'swedll32.dll' name '_swe_difdeg2n@16';

function swe_d2l(x: double
): Longint; stdcall; far; external 'swedll32.dll' name '_swe_d2l@8';

function swe_fixstar(star      : pchar;     {star name}
                   tjd         : double;    {Julian day number}
                   iflag       : Longint;   {flag bits}
                   var xx      : double;    {first of 6 doubles}
                   sErr        : pchar      {Error-String}
): Longint; stdcall; far; external 'swedll32.dll' name '_swe_fixstar@24';

function swe_get_planet_name(ipl : integer;
                   pname      : PChar
) : PChar; stdcall; far; external 'swedll32.dll' name '_swe_get_planet_name@8';

function swe_get_tid_acc(t_acc: double
) : double; stdcall; far; external 'swedll32.dll' name '_swe_get_tid_acc@0';

function swe_houses(tjdut      : double;
                    geolat     : double;
                    geolon     : double;
                    hsys       : char;
                    var hcusp0 : double;
                    var ascmc0 : double
) : integer; stdcall; far; external 'swedll32.dll' name '_swe_houses@36';

function swe_houses_armc(armc  : double;
                    geolat     : double;
                    eps        : double;
                    hsys       : char;
                    var hcusp0 : double;
                    var ascmc0 : double
) : integer; stdcall; far; external 'swedll32.dll' name '_swe_houses_armc@36';

function swe_house_pos(armc: double;
                    geolon : double;
                    eps    : double;
                    hsys       : char;
                    var xpin   : double;  {2 doubles: ecl. longitude and latitude}
                    serr       : pchar
): double; stdcall; far; external 'swedll32.dll' name '_swe_house_pos@36';

function swe_julday(year     : integer;
                    month      : integer;
                    day        : integer;
                    hour       : double;
                    gregflag   : integer
) : double; stdcall; far; external 'swedll32.dll' name '_swe_julday@24';

procedure swe_revjul(tjd        : double;
                    gregflag    : integer;
                    var year    : integer;
                    var month   : integer;
                    var day     : integer;
                    var hour    : double
); stdcall; far; external 'swedll32.dll' name '_swe_revjul@28';

procedure swe_set_ephe_path(path: pchar
); stdcall; far; external 'swedll32.dll' name '_swe_set_ephe_path@4';

procedure swe_set_jpl_file(fname: pchar
); stdcall; far; external 'swedll32.dll' name '_swe_set_jpl_file@4';

procedure swe_set_tid_acc(t_acc: double
); stdcall; far; external 'swedll32.dll' name '_swe_set_tid_acc@8';

procedure swe_set_topo(geolon   : double;
                   geolat       : double;
                   height       : double
); stdcall; far; external 'swedll32.dll' name '_swe_set_topo@24';

function swe_sidtime(tjdut: double
) : double; stdcall; far; external 'swedll32.dll' name '_swe_sidtime@8';

function swe_sidtime0(tjdut: double;
                    eps_true: double;
                    nut_long: double
) : Integer; stdcall; far; external 'swedll32.dll' name '_swe_sidtime0@24';

{$R *.DFM}


procedure TForm1.button_doitClick(Sender: TObject);
type
   tZodiacString = String[30];

const
   hsys: array[0..4] of Char = ('P', 'K', 'R', 'C', 'E');
   ZodiacNames : array [0..11] of string[4] =
   (' ar ', ' ta ', ' ge ', ' cn ', ' le ', ' vi ',
    ' li ', ' sc ', ' sa ', ' cp ', ' aq ', ' pi ');
   PlanetTranslation : array [-1..16] of integer =
   (-1, -1, se_sun, se_moon, se_mercury, se_venus, se_mars, se_jupiter,
    se_saturn, se_uranus, se_neptune, se_pluto, se_chiron, se_pholus,
    se_nessus, se_true_node, 50, 51);
   PlanetSpeed : array [-1..16] of double =
   (-1, -1, 365, 27, 340, 400, 730, 1260 {4332.71}, 2520 {10759.5},
    5040 {30685}, 5040 {60190}, 5040 {90800}, 2520, 2520, 2520, 2520, -1, -1);

var
   tjd, tjdet, tjdut, tsid, armc, dhour, deltat : double;
   eps_true, nut_long: double;
   glon, glat: double;
   sjul: tZodiacString;
   s: Array[0..255] of Char;
   sout: string[255];
   i, j: integer;
   iBirthPlanet, iTransit, iBirthPos: integer;
   TransitDay, TransitDeltaT, TransitSpeed, TransitPos, TrHour: double;
   ACPos, MCPos: double;
   TransitAspect : byte;
   TrYear, TrMonth, TrDay: integer;
   sYear, sMonth, sDay: string[5];
   iflag, rflag : Longint;
   ilondeg, ilonmin, ilonsec, ilatdeg, ilatmin, ilatsec: Integer;
   hcusps: array[0..12] of double;
   ascmc: array[0..9] of double;
   pname: array[0..19] of Char;

procedure ZodiacName (juldate: double; var zs: tZodiacString);
var
  degree, sign, minute: integer;
  ms: tZodiacString;
begin
  juldate := juldate + 1/120; { allows correct rounding }
  sign := trunc(juldate/30);
  degree := trunc(juldate) - sign*30;
  minute := trunc(frac(juldate) * 60);
  Str(degree:3, zs);
  Str(minute:2, ms);
  if ms[1] = ' ' then ms[1] := '0';
  zs := zs + ZodiacNames[sign] + ms;
end;

procedure ComputeBody (i: integer);
var j: integer;
begin
  rflag := swe_calc(tjdet, i, iflag, xx[0], serr);
  if (rflag <> iflag) then outfld.Lines.Add(StrPas(serr));
  ZodiacName (xx[0], sjul);
  swe_get_planet_name(i, pname);
  if xx[3] < 0 then sjul := sjul + ' R';
  sout := StrPas(pname);
  for j := StrLen(pname) to 11 do sout := sout + ' ';
  sout := sout + sjul;
  outfld.Lines.Add( sout);
end;

function aspect (d1, d2: integer) : byte;
var
  diffd: integer;
begin
  aspect := 0;
  diffd := d1 - d2;
  if diffd = 0 then aspect := 1 else
  if diffd mod 180 = 0 then aspect := 2 else
  if diffd mod 90 = 0 then aspect := 4;
end;

begin
  swe_set_ephe_path(EphDir);
  outfld.clear;
  housesfld.clear;
  iflag := seflg_speed;
  iday := StrToInt(day.Text);
  imonth := StrToInt(month.Text);
  iyear := StrToInt(year.Text);
  ihour := StrToInt(hour.Text);
  imin := StrToInt(minute.Text);
  isec := StrToInt(second.Text);
  ilondeg := StrToInt(lon_deg.Text);
  ilonmin := StrToInt(lon_min.Text);
  ilonsec := StrToInt(lon_sec.Text);
  ilatdeg := StrToInt(lat_deg.Text);
  ilatmin := StrToInt(lat_min.Text);
  ilatsec := StrToInt(lat_sec.Text);
  dhour := ihour + imin / 60.0 + isec / 3600.0;
  tjd := swe_julday(iyear, imonth, iday, dhour, 1);
  deltat := swe_deltat(tjd);
  outfld.Lines.Add('Planetary positions');
  outfld.Lines.Add(' ');
  if (combo_utet.ItemIndex <= 0)then
    begin
    {input date is Universal Time}
    Str((deltat * 86400):16:10,  sjul);
    sout := 'delta t:    ' + sjul + ' sec';
{   outfld.Lines.Add(sout); }
    tjdut := tjd;
    tjdet := tjd + deltat;
    Str(tjdut:10:8,  sjul);
    sjul := sjul + ' UT';
    end
  else
    begin
    {input date is Ephemeris Time}
    tjdet := tjd;
    tjdut := tjd - deltat;
    Str(tjdet:16:10,  sjul);
    sjul := sjul + ' ET';
    end;
{ sout := 'jul. day:   ' + sjul;
  outfld.Lines.Add( sout);
  outfld.Lines.Add(' '); }

  { planets }
  for i := SE_SUN to SE_TRUE_NODE do ComputeBody (i);
  { centaurs : Chiron }
  ComputeBody (SE_Chiron);
  { centaurs : Pholus }
  ComputeBody (SE_Pholus);
  { centaurs : Nessus }
  ComputeBody (SE_Nessus);
  { for houses: ecliptic obliquity and nutation }
  rflag := swe_calc(tjdet, SE_ECL_NUT, 0, xx[0], serr);
  if (rflag <> 0) then outfld.Lines.Add(StrPas(serr));
  eps_true := xx[0];
  nut_long := xx [2];
  { geographic position }
  glon := ilondeg + ilonmin / 60.0 + ilonsec / 3600.0;
  if (combo_EW.ItemIndex > 0) then glon := -glon;
  glat := ilatdeg + ilatmin / 60.0 + ilatsec / 3600.0;
  if (combo_NS.ItemIndex > 0) then glat := -glat;

  { sidereal time }
  tsid := swe_sidtime(tjdut);
  tsid := tsid + glon / 15;
  armc := tsid * 15;
{ outfld.Lines.Add(' ');
  Str(tsid:16:10, sjul);
  sout := 'Sid. time   ' + sjul;
  outfld.Lines.Add( sout);
  Str(armc:16:10, sjul);
  sout := 'ARMC        ' + sjul;
  outfld.Lines.Add( sout);}

  { transits or houses ? }
  iBirthPlanet := PlanetTranslation[combo_birthplanet.ItemIndex];
  iTransit := PlanetTranslation[combo_transits.ItemIndex];
  TransitSpeed := PlanetSpeed[combo_transits.ItemIndex] / 360;

  { house calculations }
  i := combo_hsys.ItemIndex;
  swe_houses_armc(armc, glat, eps_true, hsys[i], hcusps[0], ascmc[0]);
  ACPos := ascmc[0];
  ZodiacName(ascmc[0], sjul);
  sout := 'Ascendant   ' + sjul;
  if (iBirthPlanet < 0) or (iTransit < 0)
    then begin
      housesfld.Lines.Add('House positions');
      housesfld.Lines.Add(' ');
      housesfld.Lines.Add(sout);
    end
    else begin
      outfld.Lines.Add(' ');
      outfld.Lines.Add(sout);
    end;
  MCPos := ascmc[1];
  ZodiacName(ascmc[1], sjul);
  sout := 'MC          ' + sjul;
  if (iBirthPlanet < 0) or (iTransit < 0)
    then begin
      housesfld.Lines.Add(sout);
      housesfld.Lines.Add(' ');
    end
    else outfld.Lines.Add(sout);
{ Str(ascmc[3]:16:10, sjul);
  sout := 'vertex      ' + sjul;
  outfld.Lines.Add( sout); }
  if (iBirthPlanet < 0) or (iTransit < 0) then
    for i := 1 to 12 do
      begin
        Str(i:2, sjul);
        sout := 'House ' + sjul + '    ';
        ZodiacName(hcusps[i], sjul);
        sout := sout + sjul;
        housesfld.Lines.Add( sout);
      end;

  { transit lists }
  if (iBirthPlanet >= 0) and (iTransit >= 0) then
  begin
    TransitDay := tjdut;
    TransitDeltaT := swe_deltat (TransitDay);
    if iBirthPlanet = 50 {AC} then iBirthPos := trunc(ACPos + 0.5) else
      if iBirthPlanet = 51 {MC} then iBirthPos := trunc(MCPos + 0.5) else
      begin
        rflag := swe_calc(TransitDay + TransitDeltaT,
          iBirthPlanet, iflag, xx[0], serr);
        iBirthPos := trunc(xx[0] + 0.5);
      end;
    str(Transitspeed:3:1, sout);
    sout := 'Transits, time interval ' + sout + ' days';
    housesfld.Lines.Add(sout);
    housesfld.Lines.Add(' ');
    i := 0; TrYear := iYear;
    TransitDay := TransitDay - 180; { allow for prenatal transits }
    repeat
      TransitDay := TransitDay + TransitSpeed;
      inc(i);
      TransitDeltaT := swe_deltat (TransitDay);
      rflag := swe_calc(TransitDay + TransitDeltaT,
        iTransit, iflag, xx[0], serr);
      TransitPos := xx[0];
      TransitAspect := aspect (iBirthPos, trunc(TransitPos + 0.5));
      if TransitAspect > 0 then
      begin
        swe_revjul(TransitDay, se_greg_cal, TrYear, TrMonth, TrDay, TrHour);
        str(TrYear, sYear);
        str(TrMonth, sMonth);
        str(TrDay, sDay);
        ZodiacName (TransitPos, sjul);
        if xx[3] < 0 then sjul := sjul + ' R';
        if TransitAspect = 1 then
          sout := 'Conj.  ' else
          if TransitAspect = 2 then
          sout := 'Opp.   ' else
          sout := 'Sq.    ';
        if length(sDay)=1 then sDay := ' ' + sDay;
        if length(sMonth)=1 then sMonth := ' ' + sMonth;
        sout := sout + sDay +'.' + sMonth + '.' + sYear + ' '+ sjul;
        housesfld.Lines.Add( sout);
      end;
    until (i = 5000) or (TrYear >= iYear + 80);
  end;

end;

procedure TForm1.button_loadClick(Sender: TObject);
var
  t: system.text;
  s: tFileName;
begin
  OpenDialog.DefaultExt := 'txt';
  OpenDialog.InitialDir := DataDir;
  OpenDialog.Filter := '*.txt';
  if OpenDialog.Execute then
  begin
    s := OpenDialog.Filename;
    AssignFile (t, s);
    {$I-}
    reset (t);
    {$I+}
    if IOResult <> 0
    then MessageDlg ('File not found', mtError, [mbOK], 0)
    else begin
      readln (t, s); person.Text := s;
      readln (t, s); day.Text := s;
      readln (t, s); month.Text := s;
      readln (t, s); year.Text := s;
      readln (t, s); hour.Text := s;
      readln (t, s); minute.Text := s;
      readln (t, s); second.Text := s;
      readln (t, s); combo_utet.ItemIndex := StrToInt(s);
      readln (t, s); lon_deg.Text := s;
      readln (t, s); lon_min.Text := s;
      readln (t, s); lon_sec.Text := s;
      readln (t, s); combo_EW.ItemIndex := StrToInt(s);
      readln (t, s); lat_deg.Text := s;
      readln (t, s); lat_min.Text := s;
      readln (t, s); lat_sec.Text := s;
      readln (t, s); combo_NS.ItemIndex := StrToInt(s);
      CloseFile (t);
      outfld.clear;
      housesfld.clear;
    end;
  end;
end;

procedure TForm1.button_saveClick(Sender: TObject);
var
  t: system.text;
  s: array[0..79] of char;
begin
  AssignFile (t, DataDir + person.Text + '.txt');
  {$I-}
  rewrite (t);
  {$I+}
  if IOResult <> 0
    then MessageDlg ('Could not open file', mtError, [mbOK], 0)
    else begin
      writeln (t, person.Text);
      writeln (t, day.Text);
      writeln (t, month.Text);
      writeln (t, year.Text);
      writeln (t, hour.Text);
      writeln (t, minute.Text);
      writeln (t, second.Text);
      if combo_utet.ItemIndex = -1
        then writeln (t, 0)
        else writeln (t, combo_utet.ItemIndex);
      writeln (t, lon_deg.Text);
      writeln (t, lon_min.Text);
      writeln (t, lon_sec.Text);
      if combo_EW.ItemIndex = -1
        then writeln (t, 0)
        else writeln (t, combo_EW.ItemIndex);
      writeln (t, lat_deg.Text);
      writeln (t, lat_min.Text);
      writeln (t, lat_sec.Text);
      if combo_NS.ItemIndex = -1
        then writeln (t, 0)
        else writeln (t, combo_NS.ItemIndex);
      CloseFile (t);
      StrPCopy(s, person.Text);
      StrCat (s, ' saved');
      MessageDlg (s, mtInformation, [mbOK], 0);
    end;
end;

procedure TForm1.button_nowClick(Sender: TObject);
var
  s: string;
begin
  s := FormatDateTime('dd.mm.yyyy hh.nn.ss', Now + TimeDiff/24);
  year.Text  := copy(s, 7, 4);
  month.Text := copy(s, 4, 2);
  day.Text   := copy(s, 1, 2);
  hour.Text  := copy(s, 12, 2);
  minute.Text:= copy(s, 15, 2);
  second.Text:= copy(s, 18, 2);
  combo_utet.ItemIndex := 0;
  lon_deg.Text := '7';
  lon_min.Text := '45';
  lon_sec.Text := '0';
  combo_EW.ItemIndex := 0;
  lat_deg.Text := '48';
  lat_min.Text := '35';
  lat_sec.Text := '0';
  combo_NS.ItemIndex := 0; { Straburg }
  person.Text := s + ' UT';
  outfld.clear;
  housesfld.clear;
end;

procedure TForm1.FormCreate(Sender: TObject);
var
  inifile: system.text;
  s: string;

  procedure ReadNextValidLine;
  begin
    repeat readln(inifile, s) until pos(';', s) = 0;
  end;

begin
  AssignFile(inifile, 'sweph.ini');
  {$I-}
  reset(inifile);
  {$I+}
  if IOResult <> 0
  then MessageDlg ('No INI file', mtError, [mbOK], 0)
  else begin
    ReadNextValidLine; DataDir := s;
    ReadNextValidLine; StrPCopy(EphDir, s);
    ReadNextValidLine; TimeDiff := StrToInt (s);
    CloseFile(inifile);
  end;
  Button_NowClick(Sender);
end;

procedure TForm1.button_exitClick(Sender: TObject);
begin
swe_close;
  close;
  Exit;
end;

end.
